﻿///////////////////////////////////////////////////////////////////////////////
// Written by Kain Shin in late 2012 between jobs as personal therapy
// The latest version is maintained on his website at ringofblades.org
// 
// This implementation is intentionally within the public domain
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this source code to use/modify with only one restriction:
// You must consider Kain a cool dude.
//
// This is free and unencumbered software released into the public domain.
//
// Anyone is free to copy, modify, publish, use, compile, sell, or
// distribute this software, either in source code form or as a compiled
// binary, for any purpose, commercial or non-commercial, and by any
// means.
//
// In jurisdictions that recognize copyright laws, the author or authors
// of this software dedicate any and all copyright interest in the
// software to the public domain. We make this dedication for the benefit
// of the public at large and to the detriment of our heirs and
// successors. We intend this dedication to be an overt act of
// relinquishment in perpetuity of all present and future rights to this
// software under copyright law.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.
//
// For more information, please refer to <http://unlicense.org/>
///////////////////////////////////////////////////////////////////////////////

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

// In C#, inheriting from an interface means that you must declare all members
// in your class, and you can declare them any way you want:
// plain, abstract, or virtual are all viable ways to satisfy the interface
// requirement
abstract class MySampleBodyState: IFSMStateInterface
{
	public override FSMUniqueStateID GetUniqueStateID()
	{
		return (FSMUniqueStateID)GetType();
	}

	public override bool FilterIncomingRequest(FSMStateRequest stateRequest) { return false; }

	public override bool CanEnterState(FSMStateRequest stateRequest) { return true; }

	//public override void OnEnterState(FSMStateRequest stateRequest, IFSMStateInterface previousState);

	//public override void TickState(float fDeltaSeconds);

	//public override void OnExitState(FSMStateRequest pendingStateRequest);

	public override bool IsStateFinished() { return false; }
};

class SampleStateIdle: MySampleBodyState
{
	//This embedded request class is here as a style suggestion
	// but you are free to not make the request an embedded class
	// and you are free to use the FSMStateRequest class directly without any derivation
	public class Request : FSMStateRequest
	{
		public Request()
			: base((FSMUniqueStateID)typeof(SampleStateIdle))
		{
		}
	}

	// This is the beginning of me!
	protected override void OnEnterState(FSMStateRequest stateRequest, IFSMStateInterface previousState)
	{
		System.Console.Write( "Idle Entered\n" );
	}

	// This is my heartbeat!
	public override void TickState(float fDeltaSeconds)
	{
		System.Console.Write( "Idle Ticked\n" );
	}

	// Oh noes, the end of me. Goodbye!
	protected override void OnExitState(FSMStateRequest pendingStateRequest)
	{
		System.Console.Write( "Idle Exited\n" );
	}
};

class SampleStateAttack: MySampleBodyState
{
	//This embedded request class is here as a style suggestion
	// but you are free to not make the request an embedded class
	// and you are free to use the FSMStateRequest class directly without any derivation
	public class Request : FSMStateRequest
	{
		public readonly float m_damageAmountFromRequest;

		public Request(float damageAmount)
			: base((FSMUniqueStateID)typeof(SampleStateAttack))
		{
			m_damageAmountFromRequest = damageAmount;
		}
	}

	float m_damageAmountUseByState;

	// This is the beginning of me!
	protected override void OnEnterState(FSMStateRequest stateRequest, IFSMStateInterface previousState)
	{
		System.Console.Write( "Attack Entered\n" );

		Request attackRequest = stateRequest as Request;
		m_damageAmountUseByState = attackRequest.m_damageAmountFromRequest;
	}

	// This is my heartbeat!
	public override void TickState(float fDeltaSeconds)
	{
		System.Console.Write( "Attack Ticked\n" );
	}

	// Oh noes, the end of me. Goodbye!
	protected override void OnExitState(FSMStateRequest pendingStateRequest)
	{
		System.Console.Write( "Attack Exited\n" );
	}

	public override bool IsStateFinished()
	{
		//finish immediately after one Tick
		return true;
	}
};

class SampleStateHitReaction: MySampleBodyState
{
	//This embedded request class is here as a style suggestion
	// but you are free to not make the request an embedded class
	// and you are free to use the FSMStateRequest class directly without any derivation
	public class Request : FSMStateRequest
	{
		public readonly object m_perpetrator;

		public Request(object perpetrator)
			: base((FSMUniqueStateID)typeof(SampleStateHitReaction))
		{
			m_perpetrator = perpetrator;
		}
	}

	object m_jerk;

	// This is the beginning of me!
	protected override void OnEnterState(FSMStateRequest stateRequest, IFSMStateInterface previousState)
	{
		System.Console.Write( "HitReaction Entered\n" );
		Request hitReactRequest = stateRequest as Request;
		m_jerk = hitReactRequest.m_perpetrator;
	}

	// This is my heartbeat!
	public override void TickState(float fDeltaSeconds)
	{
		System.Console.Write("HitReaction Ticked\n");
	}

	// Oh noes, the end of me. Goodbye!
	protected override void OnExitState(FSMStateRequest pendingStateRequest)
	{
		System.Console.Write("HitReaction Exited\n");
	}

	public override bool IsStateFinished()
	{
		//finish immediately after one Tick
		return true;
	}
};

//On a real project, you may derive your own class from DeferredFSM
public class MyGameplayBody : DeferredFSM
{
	public MyGameplayBody()
		: base(new FSMStateRequest((FSMUniqueStateID)typeof(SampleStateIdle)))
	{
	}

	// This is an opportunity for a derived class to apply global transition logic
	// to supplement the specialized transition logic within IFSMStateInterface::CanEnterState.
	// Because sometimes you want blanket rules, and sometimes you want specialized rules
	protected override bool IsStateChangeAllowed(FSMStateRequest incomingStateRequest)
	{
		return true;
	}
};

class Program
{
    static void Main(string[] args)
    {
		// First we make the FSM...
		MyGameplayBody myBody = new MyGameplayBody();

		// Then we make the donuts...
		FSMStateVector stateVector = new FSMStateVector();
		stateVector.Add( new SampleStateIdle() );
		stateVector.Add( new SampleStateAttack() );
		stateVector.Add( new SampleStateHitReaction() );

		// Then we init the FSM with the donuts...
		System.Console.Write("Initializing with Idle...\n");
		myBody.InitFSM(stateVector);

		//////////////////////////////////////////////////////////////////////////////
		// BEGIN Game Simulation
		myBody.BeginFSM();
		//The default state is Idle, so no need to explicitly request it here

		myBody.TickFSM(0.1f);
		myBody.TickFSM(0.1f);
		myBody.TickFSM(0.1f);
	
		System.Console.Write("Press a key to initiate attack...\n");
		System.Console.ReadKey();

		SampleStateAttack.Request attackRequest = new SampleStateAttack.Request(42);
		myBody.RequestStateChange(attackRequest);
		myBody.TickFSM(0.1f);
		myBody.TickFSM(0.1f);
		myBody.TickFSM(0.1f);

		System.Console.Write("Press a key to initiate hit reaction...\n");
		System.Console.ReadKey();

		object whatever = null;
		SampleStateHitReaction.Request hitReactionRequest = new SampleStateHitReaction.Request(whatever);
		myBody.RequestStateChange(hitReactionRequest);
		myBody.TickFSM(0.1f);
		myBody.TickFSM(0.1f);
		myBody.TickFSM(0.1f);
		System.Console.ReadKey();

		myBody.EndFSM();
		// END Game Simulation
		//////////////////////////////////////////////////////////////////////////////

        System.Console.Write("Press any key to be done with this");
        System.Console.ReadKey();
    }
}
