///////////////////////////////////////////////////////////////////////////////
// Converted from a C++ version that was
// Written by Kain Shin in late 2012 between jobs as personal therapy
// The latest version is maintained on his website at ringofblades.org
// 
// This implementation is intentionally within the public domain
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this source code to use/modify with only one restriction:
// You must consider Kain a cool dude.
//
// This is free and unencumbered software released into the public domain.
//
// Anyone is free to copy, modify, publish, use, compile, sell, or
// distribute this software, either in source code form or as a compiled
// binary, for any purpose, commercial or non-commercial, and by any
// means.
//
// In jurisdictions that recognize copyright laws, the author or authors
// of this software dedicate any and all copyright interest in the
// software to the public domain. We make this dedication for the benefit
// of the public at large and to the detriment of our heirs and
// successors. We intend this dedication to be an overt act of
// relinquishment in perpetuity of all present and future rights to this
// software under copyright law.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.
//
// For more information, please refer to <http://unlicense.org/>
///////////////////////////////////////////////////////////////////////////////

using System.Diagnostics;

///////////////////////////////////////////////////////////////////////////////
// Basic types are declared here to keep usage intentions defined at a single place
public class FSMStateVector : System.Collections.Generic.List<IFSMStateInterface>
{
	// C# does not do typedefs that transcend files
}

public class FSMStateMap : System.Collections.Generic.Dictionary<FSMUniqueStateID, IFSMStateInterface>
{
	// C# does not do typedefs that transcend files
}

public class FSMStateRequestQueue : System.Collections.Generic.List<FSMStateRequest>
{
	// C# does not do typedefs that transcend files
}
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// This class represents a DEFERRED finite state machine
// Deferred state machines have the downside of having to deal with
// "pending state" ambiguity when you want to know "current state" at a
// certain snapshot in time but the upside is that your code would be highly
// resistant to tick order dependencies.
// Tick order bugs can be the hardest bugs to get 100% repro on when you
// have numerous inter-dependent participants being updated within the same
// 'forloop'
// It is assumed that you will derive from this class with your "GameCharacter" class
public class DeferredFSM
{
#if UNITY_EDITOR
	///////////////////////////////////////////////////////////////////////////////
	[Conditional("UNITY_EDITOR")]
	public static void ENGINE_SPECIFIC_ASSERT(bool bTest, string assertMsg)
	{
		if (!bTest)
		{
			UnityEngine.Debug.LogError(assertMsg);
		}
	}
#else
	///////////////////////////////////////////////////////////////////////////////
	//Unity users should just delete this section and depend solely
	// on the Conditional keyword used above
	public static void ENGINE_SPECIFIC_ASSERT(bool bTest, string assertMsg)
	{
		Debug.Assert(bTest, assertMsg);
	}
#endif //UNITY_EDITOR

	///////////////////////////////////////////////////////////////////////////////
	private readonly FSMStateRequest m_DefaultStateRequest;
	protected IFSMStateInterface m_pCurrentState = null;
	private FSMStateMap m_StateMap = new FSMStateMap();

	//State requests may be made during processing inside OnEnterState or OnExitState
	//For this reason, we "page-flip" to ensure that the currently processed queue
	//is not disturbed during any state transition
	const int kNumStateRequestQueues = 2;
	private FSMStateRequestQueue[] m_pendingStateRequests = { new FSMStateRequestQueue(), new FSMStateRequestQueue() };
	int m_activeStateRequestQueueIndex = 0;

	///////////////////////////////////////////////////////////////////////////////
	public DeferredFSM(FSMStateRequest defaultStateRequest)
	{
		m_DefaultStateRequest = defaultStateRequest;
	}

	///////////////////////////////////////////////////////////////////////////////
	~DeferredFSM()
	{
		//Note that m_StateMap was merely holding pointers that were being managed
		// by whoever passed that FSMStateVector to the constructor
	}

	///////////////////////////////////////////////////////////////////////////////
	// The data is known...
	public void InitFSM(FSMStateVector finiteStates)
	{
		// InitFSM was coded to only be called once per object lifetime
		// to avoid any memory jiggling associated calling m_StateMap.clear()
		// and then filling the map back up again with the same exact stuff
		ENGINE_SPECIFIC_ASSERT(0 == m_StateMap.Count, "Hey programmer, InitFSM only needs to be called once per object to avoid jiggling memory");

		foreach (IFSMStateInterface finiteState in finiteStates)
		{
			FSMUniqueStateID uniqueStateID = finiteState.GetUniqueStateID();
			ENGINE_SPECIFIC_ASSERT(!m_StateMap.ContainsKey(uniqueStateID), "Hey programmer, it looks like you might be trying to insert two states into the machine that share the same ID");
			m_StateMap.Add(uniqueStateID, finiteState);
		}

		ENGINE_SPECIFIC_ASSERT(m_StateMap.ContainsKey(m_DefaultStateRequest.GetRequestID()), "Hey programmer, make sure your default state is legal, mmmkay?");

		foreach (FSMStateRequestQueue stateRequestQueue in m_pendingStateRequests)
		{
			stateRequestQueue.Clear();
		}
		m_activeStateRequestQueueIndex = 0;
	}

	///////////////////////////////////////////////////////////////////////////////
	// The game has begun...
	public void BeginFSM()
	{
		RequestStateChange(m_DefaultStateRequest);
	}

	///////////////////////////////////////////////////////////////////////////////
	// This request returns void by design because the result of this request
	// will not be known until the next Tick. Conditional state queries before
	// making the request would alleviate the issue of not knowing, but those
	// can lie to you if the state change is possible in this frame but NOT in
	// the next frame.
	// You are highly encouraged to code asynchronously with this DeferredFSM
	// because rules for state transitions are not consistent between frames
	public void RequestStateChange(FSMStateRequest incomingStateRequest)
	{
		ENGINE_SPECIFIC_ASSERT(m_StateMap.ContainsKey(incomingStateRequest.GetRequestID()), "Somehow, a state was requested that does not exist within the machine");

		FSMStateRequestQueue activeStateRequestQueue = m_pendingStateRequests[m_activeStateRequestQueueIndex];

		bool bRequestPending = true;
		FSMStateRequestQueue pendingRequestsToRemoveIfNewGuySurvives = new FSMStateRequestQueue();
		int numPendingStateRequests = activeStateRequestQueue.Count;
		FSMStateRequest pendingStateRequest;
		FSMStateRequest doubleSuicideRequest = null;
		for (int i=0; bRequestPending && (i < numPendingStateRequests); ++i)
		{
			// Each incoming pending request must pass the gauntlet of all existing pending requests
			// just like the new guy in prison hoping to one day make it as the "current state"
			pendingStateRequest = activeStateRequestQueue[i];
			switch (incomingStateRequest.VersusOtherRequest(pendingStateRequest))
			{
				case FSMStateRequest.VersusResult.VersusResult_BothSurvive:
					break;
				case FSMStateRequest.VersusResult.VersusResult_CancelSelf:
					bRequestPending = false;
					break;
				case FSMStateRequest.VersusResult.VersusResult_CancelOther:
					pendingRequestsToRemoveIfNewGuySurvives.Add(pendingStateRequest);
					break;
				case FSMStateRequest.VersusResult.VersusResult_CancelBoth:
					bRequestPending = false;
					doubleSuicideRequest = pendingStateRequest;
					break;
			}
		}

		if (bRequestPending)
		{
			foreach (FSMStateRequest doomedStateRequest in pendingRequestsToRemoveIfNewGuySurvives)
			{
				activeStateRequestQueue.Remove(doomedStateRequest);
			}

			activeStateRequestQueue.Add(incomingStateRequest);
		}
		else if (null != doubleSuicideRequest)
		{
			activeStateRequestQueue.Remove(doubleSuicideRequest);
		}
	}

	///////////////////////////////////////////////////////////////////////////////
	// BEWARE: There is no guarantee that this current state will be the current state in the next frame
	// Use this only when timing does not matter
	public FSMUniqueStateID GetCurrentStateID()
	{
		FSMUniqueStateID currentStateID = (null != m_pCurrentState)
											? m_pCurrentState.GetUniqueStateID()
											: m_DefaultStateRequest.GetRequestID();
		return currentStateID;
	}

	///////////////////////////////////////////////////////////////////////////////
	// BEWARE: There is no guarantee that a pending state will actually become current
	// Use this only when precision does not matter
	// LOGIC BASED ON PENDING STATES CAN LEAD TO SUBTLE BUGS
	// THIS FUNCTION IS LEFT COMMENTED TO ACKNOWLEDGE YOUR DESIRE AND WARN
	// YOU TO RESIST THE TEMPTATION TO USE PENDING STATE IN YOUR LOGIC
	//bool IsStateCurrentOrPending( FSMUniqueStateID const uniqueStateID ) const;

	///////////////////////////////////////////////////////////////////////////////
	// Because sometimes, you just want to read data from a state
	public IFSMStateInterface GetStateForQuery(FSMUniqueStateID uniqueStateID)
	{
		IFSMStateInterface queriedState = null;
		m_StateMap.TryGetValue(uniqueStateID, out queriedState);
		ENGINE_SPECIFIC_ASSERT(null != queriedState, "Hey programmer, you requested a state that has not been added to the state machine.");
		return queriedState;
	}

	///////////////////////////////////////////////////////////////////////////////
	public void TickFSM(float fDeltaSeconds)
	{
		ProcessPendingStates();

		if (null != m_pCurrentState)
		{
			if (m_pCurrentState.IsStateFinished())
			{
				IFSMStateInterface pFinishedState = m_pCurrentState;
				RequestStateChange(m_DefaultStateRequest);
				ProcessPendingStates();
				ENGINE_SPECIFIC_ASSERT(pFinishedState != m_pCurrentState, "Hey programmer, IsStateFinished() happened, but all transitions got denied");
				if (null != m_pCurrentState)
				{
					m_pCurrentState.TickState(fDeltaSeconds);
				}
			}
			else
			{
				m_pCurrentState.TickState(fDeltaSeconds);
			}
		}
	}

	///////////////////////////////////////////////////////////////////////////////
	// The game has ended...
	public void EndFSM()
	{
		if (null != m_pCurrentState)
		{
			m_pCurrentState.DoExitState(null);
			m_pCurrentState = null;
		}

		foreach (FSMStateRequestQueue stateRequestQueue in m_pendingStateRequests)
		{
			stateRequestQueue.Clear();
		}
	}

	///////////////////////////////////////////////////////////////////////////////
	// This is an opportunity for a derived class to apply global transition logic
	// to supplement the specialized transition logic within IFSMStateInterface::CanEnterState.
	// Because sometimes you want blanket rules, and sometimes you want specialized rules
	protected virtual bool IsStateChangeAllowed(FSMStateRequest incomingStateRequest)
	{
		return true;
	}

	///////////////////////////////////////////////////////////////////////////////
	private void ProcessPendingStates()
	{
		FSMStateRequestQueue processingStateRequestQueue = m_pendingStateRequests[m_activeStateRequestQueueIndex];

		//Make sure all subsequent state transitions go to another queue
		m_activeStateRequestQueueIndex = (1 + m_activeStateRequestQueueIndex) % kNumStateRequestQueues;

		foreach (FSMStateRequest pendingStateRequest in processingStateRequestQueue)
		{
			// 1. Ask the owner if this is allowed...
			bool bRequestGranted = IsStateChangeAllowed(pendingStateRequest);
			if (bRequestGranted && (null != m_pCurrentState))
			{
				// 2. Next, we ask the current state if this is allowed
				bRequestGranted = m_pCurrentState.IsStateFinished() || !m_pCurrentState.FilterIncomingRequest(pendingStateRequest);
			}
			if (bRequestGranted)
			{
				IFSMStateInterface pendingState = null;

				// There is an assert within DeferredFSM::RequestStateChange that ensures that
				// we have a legal state ID
				m_StateMap.TryGetValue(pendingStateRequest.GetRequestID(), out pendingState);
				ENGINE_SPECIFIC_ASSERT(null != pendingState, "Hey programmer, you requested a state that has not been added to the state machine.");

				// 3. Now the state itself gets a chance to be allowed
				bRequestGranted = pendingState.CanEnterState(pendingStateRequest);
				if (bRequestGranted)
				{
					// Do the state transition HERE...
					if (null != m_pCurrentState)
					{
						m_pCurrentState.DoExitState(pendingStateRequest);
					}
					ENGINE_SPECIFIC_ASSERT(pendingState.GetUniqueStateID() == pendingStateRequest.GetRequestID(), "Hey programmer, a promise for unique state ID to match the pending state has somehow been broken");
					pendingState.DoEnterState(pendingStateRequest, m_pCurrentState);
					m_pCurrentState = pendingState;
				}
			}

			// Note from Kain: I thought about supporting the concept of "Lingering Requests"
			// where a request would stick around for X seconds trying to become current
			// while it lingered, but then I thought about the prospect of hard to repro
			// bugs that might pop up because a request from X seconds ago lingered and
			// manifested at some inopportune time... and so I backed out because I figured
			// the reward of lingering state requests would not outweigh the risk.
			// Linger state requests should be done by having calling code actively manage that intention
			// instead of doing a fire-and-forget usage pattern
		}

		processingStateRequestQueue.Clear();
	}
};