///////////////////////////////////////////////////////////////////////////////
// Written by Kain Shin in late 2012 between jobs as personal therapy
// The latest version is maintained on his website at ringofblades.org
// 
// This implementation is intentionally within the public domain
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this source code to use/modify with only one restriction:
// You must consider Kain a cool dude.
//
// This is free and unencumbered software released into the public domain.
//
// Anyone is free to copy, modify, publish, use, compile, sell, or
// distribute this software, either in source code form or as a compiled
// binary, for any purpose, commercial or non-commercial, and by any
// means.
//
// In jurisdictions that recognize copyright laws, the author or authors
// of this software dedicate any and all copyright interest in the
// software to the public domain. We make this dedication for the benefit
// of the public at large and to the detriment of our heirs and
// successors. We intend this dedication to be an overt act of
// relinquishment in perpetuity of all present and future rights to this
// software under copyright law.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.
//
// For more information, please refer to <http://unlicense.org/>
///////////////////////////////////////////////////////////////////////////////


#include "GlobalIncludes.h"
#include <stdio.h>
#include <conio.h>

#include "DeferredFSM.h"

class MySampleBodyState: public IFSMStateInterface
{
public:
	//virtual FSMUniqueStateID GetUniqueStateID() const = 0;

	virtual bool FilterIncomingRequest( FSMStateRequest const& rStateRequest ) { return false; }

	virtual bool CanEnterState( FSMStateRequest const& rStateRequest ) const { return true; }

	//virtual void OnEnterState( FSMStateRequest const& rStateRequest, IFSMStateInterface* const pPreviousState ) = 0;

	//virtual void TickState( float const fDeltaSeconds ) = 0;

	//virtual void OnExitState( FSMStateRequest const* const pPendingStateRequest ) = 0;

	virtual bool IsStateFinished() const { return false; }
};

class SampleStateIdle: public MySampleBodyState
{
public:
	//This embedded request class is here as a style suggestion
	// but you are free to not make the request an embedded class
	// and you are free to use the FSMStateRequest class directly without any derivation
	struct Request : FSMStateRequest
	{
		Request()
			: FSMStateRequest(GetUniqueClassID())
		{
		}
	};

	static FSMUniqueStateID GetUniqueClassID()
	{
		return &GetUniqueClassID;
	}
	virtual FSMUniqueStateID GetUniqueStateID() const
	{
		return GetUniqueClassID();
	}

	// This is the beginning of me!
	virtual void OnEnterState( FSMStateRequest const& rStateRequest, IFSMStateInterface* const pPreviousState )
	{
		printf( "Idle Entered\n" );
	}

	// This is my heartbeat!
	virtual void TickState( float const fDeltaSeconds )
	{
		printf( "Idle Ticked\n" );
	}

	// Oh noes, the end of me. Goodbye!
	virtual void OnExitState( FSMStateRequest const* const pPendingStateRequest )
	{
		printf( "Idle Exited\n" );
	}
};

class SampleStateAttack: public MySampleBodyState
{
public:
	//This embedded request class is here as a style suggestion
	// but you are free to not make the request an embedded class
	// and you are free to use the FSMStateRequest class directly without any derivation
	struct Request : FSMStateRequest
	{
		float const m_damageAmountFromRequest;

		Request(float damageAmount)
			: FSMStateRequest(GetUniqueClassID())
			, m_damageAmountFromRequest(damageAmount)
		{
		}
	};

private:
	float m_damageAmountUseByState;

public:
	static FSMUniqueStateID GetUniqueClassID()
	{
		return &GetUniqueClassID;
	}
	virtual FSMUniqueStateID GetUniqueStateID() const
	{
		return GetUniqueClassID();
	}

	// This is the beginning of me!
	virtual void OnEnterState( FSMStateRequest const& rStateRequest, IFSMStateInterface* const pPreviousState )
	{
		printf( "Attack Entered\n" );

		Request const& rAttackRequest = static_cast<Request const&>(rStateRequest);
		m_damageAmountUseByState = rAttackRequest.m_damageAmountFromRequest;
	}

	// This is my heartbeat!
	virtual void TickState( float const fDeltaSeconds )
	{
		printf( "Attack Ticked\n" );
	}

	// Oh noes, the end of me. Goodbye!
	virtual void OnExitState( FSMStateRequest const* const pPendingStateRequest )
	{
		printf( "Attack Exited\n" );
	}

	virtual bool IsStateFinished() const
	{
		//finish immediately after one Tick
		return true;
	}
};

class SampleStateHitReaction: public MySampleBodyState
{
public:
	//This embedded request class is here as a style suggestion
	// but you are free to not make the request an embedded class
	// and you are free to use the FSMStateRequest class directly without any derivation
	struct Request : FSMStateRequest
	{
		void* const m_perpetrator;

		Request(void* const perpetrator)
			: FSMStateRequest(GetUniqueClassID())
			, m_perpetrator(perpetrator)
		{
		}
	};

private:
	void* m_jerk;

public:
	static FSMUniqueStateID GetUniqueClassID()
	{
		return &GetUniqueClassID;
	}
	virtual FSMUniqueStateID GetUniqueStateID() const
	{
		return GetUniqueClassID();
	}

	// This is the beginning of me!
	virtual void OnEnterState( FSMStateRequest const& rStateRequest, IFSMStateInterface* const pPreviousState )
	{
		printf( "HitReaction Entered\n" );

		Request const& rHitReactRequest = static_cast<Request const&>(rStateRequest);
		m_jerk = rHitReactRequest.m_perpetrator;
	}

	// This is my heartbeat!
	virtual void TickState( float const fDeltaSeconds )
	{
		printf( "HitReaction Ticked\n" );
	}

	// Oh noes, the end of me. Goodbye!
	virtual void OnExitState( FSMStateRequest const* const pPendingStateRequest )
	{
		printf( "HitReaction Exited\n" );
	}

	virtual bool IsStateFinished() const
	{
		//finish immediately after one Tick
		return true;
	}
};

//In a more complex game, a body may have several FSMs for
// full body vs. upper or left/right hand and head, etc
// We simplify body concept for the sake of this example
class MyGameplayBody : public DeferredFSM
{
private:
	SampleStateIdle* const m_pIdleState;
	SampleStateAttack* const m_pAttackState;
	SampleStateHitReaction* const m_pHitReactionState;

public:
	MyGameplayBody()
		: DeferredFSM()
		, m_pIdleState(new SampleStateIdle())
		, m_pAttackState(new SampleStateAttack())
		, m_pHitReactionState(new SampleStateHitReaction())
	{
		// Then we make the donuts...
		FSMStateVector stateVector;
		stateVector.push_back(m_pIdleState);
		stateVector.push_back(m_pAttackState);
		stateVector.push_back(m_pHitReactionState);

		// Then we init the FSM with the donuts...
		printf("Initializing with Idle...\n");
		InitFSM(stateVector);
	}

	~MyGameplayBody()
	{
		// Clean up the donuts...
		delete m_pIdleState;
		delete m_pAttackState;
		delete m_pHitReactionState;
	}

	virtual void RequestDefaultState()
	{
		RequestStateChange(STATEREQUEST_MEMORY_POOL_NEW(SampleStateIdle::Request()));
	}
	
	virtual FSMUniqueStateID GetDefaultRequestID() const
	{
		return SampleStateIdle::GetUniqueClassID();
	}

	// This is an opportunity for a derived class to apply global transition logic
	// to supplement the specialized transition logic within IFSMStateInterface::CanEnterState.
	// Because sometimes you want blanket rules, and sometimes you want specialized rules
	virtual bool IsStateChangeAllowed( FSMStateRequest const& rIncomingStateRequest ) const
	{
		return true;
	}
};

int main(int argc, CHAR* argv[])
{
	// First we make the FSM...
	MyGameplayBody myBody;

	//////////////////////////////////////////////////////////////////////////////
	// BEGIN Game Simulation
	myBody.BeginFSM();
	//The default state is Idle, so no need to explicitly request it here
	
	myBody.TickFSM( 0.1f );
	myBody.TickFSM( 0.1f );
	myBody.TickFSM( 0.1f );

	printf( "Press a key to initiate attack...\n" );
	_getch();

	SampleStateAttack::Request* pAttackRequest = STATEREQUEST_MEMORY_POOL_NEW(SampleStateAttack::Request(42));
	myBody.RequestStateChange(pAttackRequest);
	myBody.TickFSM( 0.1f );
	myBody.TickFSM( 0.1f );
	myBody.TickFSM( 0.1f );

	printf( "Press a key to initiate hit reaction...\n" );
	_getch();

	SampleStateHitReaction::Request* pHitReactionRequest = STATEREQUEST_MEMORY_POOL_NEW(SampleStateHitReaction::Request(NULL));
	myBody.RequestStateChange(pHitReactionRequest);
	myBody.TickFSM( 0.1f );
	myBody.TickFSM( 0.1f );
	myBody.TickFSM( 0.1f );
	_getch();

	myBody.EndFSM();
	// END Game Simulation
	//////////////////////////////////////////////////////////////////////////////

	printf( "Press any key to be done with this" );
	_getch();
	return 0;
}

