///////////////////////////////////////////////////////////////////////////////
// Written by Kain Shin in late 2012 between jobs as personal therapy
// The latest version is maintained on his website at ringofblades.org
// 
// This implementation is intentionally within the public domain
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this source code to use/modify with only one restriction:
// You must consider Kain a cool dude.
//
// This is free and unencumbered software released into the public domain.
//
// Anyone is free to copy, modify, publish, use, compile, sell, or
// distribute this software, either in source code form or as a compiled
// binary, for any purpose, commercial or non-commercial, and by any
// means.
//
// In jurisdictions that recognize copyright laws, the author or authors
// of this software dedicate any and all copyright interest in the
// software to the public domain. We make this dedication for the benefit
// of the public at large and to the detriment of our heirs and
// successors. We intend this dedication to be an overt act of
// relinquishment in perpetuity of all present and future rights to this
// software under copyright law.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.
//
// For more information, please refer to <http://unlicense.org/>
///////////////////////////////////////////////////////////////////////////////


#ifndef FSM_STATE_H
#define FSM_STATE_H

///////////////////////////////////////////////////////////////////////////////
//Replace this with whatever you end up using in your engine
//
// One class per state is highly recommended because trying to reuse the same
// class to represent many states (i.e. GenericSwissArmyState) will result
// in needing to support state transition intentions within the states inside that state
// and that means now supporting two models of transition logic, one within your state and one
// with this FSM framework. It's doable and sometimes neccesary to go this route,
// but do be careful if you make GenericSwissArmyState (i.e. GenericAnim or DeathState)
// If you find yourself needing to diverge state transition logic, then it may be
// time to derive a new State class from some kind of common interface
//
// Many engines have access to a StaticGetClass() or StaticGetClassName() method which eliminates
// any need for an enum as long as you have one class representing one state
// If divergent usage requirements arise between ints and hashstrings, then
// this stuff can be wrapped inside derived template classes where T is the ID
// type, but I'd rather not do that unless absolutely neccessary in order to
// limit complexity and codebloat
typedef void* FSMUniqueStateID; //If serialization is not an issue, you can use the address of a static class member variable or function
//typedef int FSMUniqueStateID; //enums assume you will static cast as needed ...
//#include <string>
//typedef std::string FSMUniqueStateID; //ClassName hashstring or whateves...
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// Derive from this struct to be able to send a bag of data with your request
// To become CurrentState a StateRequest must pass through the following Gauntlet:
// 0) FSMStateRequest::VersusOtherRequest - the other requests in purgatory will do battle with the newcomer
// 1) DeferredFSM::IsStateChangeAllowed - The owner of states will have veto powers on any state transition. Good for allowing death to veto
// 2) IFSMStateInterface::FilterIncomingRequest - The current state gets a chance to protect itself from interruptions by another state as well as handle responses to incoming requests
// 3) IFSMStateInterface::CanEnterState - Finally, the state itself will get a choice to evaluate whether it is worthy
struct FSMStateRequest
{
	explicit FSMStateRequest( FSMUniqueStateID const uniqueStateID );

	FSMUniqueStateID GetRequestID() const;

	enum VersusResult
	{
		VersusResult_BothSurvive,
		VersusResult_CancelSelf,
		VersusResult_CancelOther,
		VersusResult_CancelBoth
	};

	// A derived struct may wish to compare derived members instead of
	// making a blanket decision to take the most recent redundant request
	virtual VersusResult VersusOtherRequest( FSMStateRequest const& rOtherRequest ) const;

private:
	FSMUniqueStateID m_uniqueStateID;
};
 

///////////////////////////////////////////////////////////////////////////////
// This is intended to be a pure abstract interface class
// Please keep this base class clean and generic enough to be used by
// different code systems in this game and future games to come
class IFSMStateInterface
{
private:
	// This is the beginning of me!
	virtual void OnEnterState(FSMStateRequest const& rStateRequest, IFSMStateInterface* const pPreviousState) = 0;
	// Oh noes, the end of me. Goodbye!
	virtual void OnExitState(FSMStateRequest const* const pPendingStateRequest) = 0;

	bool m_isStateActive = false;

public:
	// This is what makes me a mutually exclusive snowflake against other snowflakes
	virtual FSMUniqueStateID GetUniqueStateID() const = 0;

	// Return true to Filter (block) an incoming stateRequest and prevent it from causing a future state change
	// This is also a good place to do things within the self without changing state
	// such as hit reactions on a dead ragdolled body
	// or an attack request while attack is currently active to initiate some combo string of chaining attacks
	virtual bool FilterIncomingRequest( FSMStateRequest const& rStateRequest ) = 0;

	// Am I allowed to be requested as the current state at this time?
	virtual bool CanEnterState( FSMStateRequest const& rStateRequest ) const = 0;

	void DoEnterState(FSMStateRequest const& rStateRequest, IFSMStateInterface* const pPreviousState);

	// This is my heartbeat!
	virtual void TickState( float const fDeltaSeconds ) = 0;

	void DoExitState(FSMStateRequest const* const pPendingStateRequest);

	bool IsStateActive() const;

	// Am I done yet? If so, then it is time to revert to the default state of the FSM
	virtual bool IsStateFinished() const = 0;
};

#endif //FSM_STATE_H