///////////////////////////////////////////////////////////////////////////////
// Written by Kain Shin in late 2012 between jobs as personal therapy
// The latest version is maintained on his website at ringofblades.org
// 
// This implementation is intentionally within the public domain
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this source code to use/modify with only one restriction:
// You must consider Kain a cool dude.
//
// This is free and unencumbered software released into the public domain.
//
// Anyone is free to copy, modify, publish, use, compile, sell, or
// distribute this software, either in source code form or as a compiled
// binary, for any purpose, commercial or non-commercial, and by any
// means.
//
// In jurisdictions that recognize copyright laws, the author or authors
// of this software dedicate any and all copyright interest in the
// software to the public domain. We make this dedication for the benefit
// of the public at large and to the detriment of our heirs and
// successors. We intend this dedication to be an overt act of
// relinquishment in perpetuity of all present and future rights to this
// software under copyright law.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.
//
// For more information, please refer to <http://unlicense.org/>
///////////////////////////////////////////////////////////////////////////////

#ifndef DEFERRED_FSM_H
#define DEFERRED_FSM_H

#include "FSMState.h"

///////////////////////////////////////////////////////////////////////////////
//Replace this with whatever you end up using in your engine
#include <vector>
typedef std::vector<IFSMStateInterface*> FSMStateVector;

#include <map>
typedef std::map<FSMUniqueStateID const, IFSMStateInterface* const> FSMStateMap;

// Minimize dynamic allocations for every request made by using a pool
//FSMStateRequest will be a base class, and so it won't be safe to have
// a container of non-pointer objects since derived classes will have different sizes
// This means that you may want to use a pool allocator for your StateRequests
// to prevent fragmentation upon frequent state requests during runtime
#define STATEREQUEST_MEMORY_POOL_NEW(p) new p
#define STATEREQUEST_MEMORY_POOL_DELETE(p) delete p
#include <list>
typedef std::list<FSMStateRequest const*> FSMStateRequestQueue;

///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// This class represents a DEFERRED finite state machine
// Deferred state machines have the downside of having to deal with
// "pending state" ambiguity when you want to know "current state" at a
// certain snapshot in time but the upside is that your code would be highly
// resistant to tick order dependencies.
// Tick order bugs can be the hardest bugs to get 100% repro on when you
// have numerous inter-dependent participants being updated within the same
// 'forloop'
// It is assumed that you will derive from this class with your "GameCharacter" class
class DeferredFSM
{
public:
	explicit DeferredFSM();
	virtual ~DeferredFSM();

	// The data is known...
	void InitFSM( FSMStateVector const& rFiniteStates );

	// The game has begun...
	void BeginFSM();

	// This request returns void by design because the result of this request
	// will not be known until the next Tick. Conditional state queries before
	// making the request would alleviate the issue of not knowing, but those
	// can lie to you if the state change is possible in this frame but NOT in
	// the next frame.
	// You are highly encouraged to code asynchronously with this DeferredFSM
	// because rules for state transitions are not consistent between frames
	void RequestStateChange( FSMStateRequest const* const pIncomingStateRequest );

	// BEWARE: There is no guarantee that this current state will be the current state in the next frame
	// Use this only when timing does not matter
	FSMUniqueStateID GetCurrentStateID() const;
	
	virtual void RequestDefaultState() = 0;
	virtual FSMUniqueStateID GetDefaultRequestID() const = 0;

	// BEWARE: There is no guarantee that a pending state will actually become current
	// Use this only when precision does not matter
	// LOGIC BASED ON PENDING STATES IS CAN LEAD TO SUBTLE BUGS
	// THIS FUNCTION IS LEFT COMMENTED TO ACKNOWLEDGE YOUR DESIRE AND WARN
	// YOU TO RESIST THE TEMPTATION TO USE PENDING STATE IN YOUR LOGIC
	//bool IsStateCurrentOrPending( FSMUniqueStateID const uniqueStateID ) const;

	// Because sometimes, you just want to read data from a state
	IFSMStateInterface const* const GetStateForQuery( FSMUniqueStateID const uniqueStateID ) const;

	void TickFSM( float const fDeltaSeconds );

	// The game has ended...
	void EndFSM();

private:
	// This is an opportunity for a derived class to apply global transition logic
	// to supplement the specialized transition logic within IFSMStateInterface::CanEnterState.
	// Because sometimes you want blanket rules, and sometimes you want specialized rules
	virtual bool IsStateChangeAllowed( FSMStateRequest const& rIncomingStateRequest ) const
	{
		return true;
	}

	void ProcessPendingStates();

	FSMStateMap m_stateMap;

	//State requests may be made during processing inside OnEnterState or OnExitState
	//For this reason, we "page-flip" to ensure that the currently processed queue
	//is not disturbed during any state transition
	static const int kNumStateRequestQueues = 2;
	
	FSMStateRequestQueue m_pendingStateRequests[kNumStateRequestQueues];
	
	int m_activeStateRequestQueueIndex = 0;

protected:
	IFSMStateInterface* m_pCurrentState;
};


#endif //DEFERRED_FSM_H
