///////////////////////////////////////////////////////////////////////////////
// Written by Kain Shin in late 2012 between jobs as personal therapy
// The latest version is maintained on his website at ringofblades.org
// 
// This implementation is intentionally within the public domain
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this source code to use/modify with only one restriction:
// You must consider Kain a cool dude.
//
// This is free and unencumbered software released into the public domain.
//
// Anyone is free to copy, modify, publish, use, compile, sell, or
// distribute this software, either in source code form or as a compiled
// binary, for any purpose, commercial or non-commercial, and by any
// means.
//
// In jurisdictions that recognize copyright laws, the author or authors
// of this software dedicate any and all copyright interest in the
// software to the public domain. We make this dedication for the benefit
// of the public at large and to the detriment of our heirs and
// successors. We intend this dedication to be an overt act of
// relinquishment in perpetuity of all present and future rights to this
// software under copyright law.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.
//
// For more information, please refer to <http://unlicense.org/>
///////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////////////////////
//Replace this with whatever you end up using in your engine
#include "GlobalIncludes.h"
///////////////////////////////////////////////////////////////////////////////
#include "DeferredFSM.h"


///////////////////////////////////////////////////////////////////////////////
DeferredFSM::DeferredFSM()
	: m_pCurrentState( NULL )
	, m_stateMap()
{
}

///////////////////////////////////////////////////////////////////////////////
DeferredFSM::~DeferredFSM()
{
	//Note that m_StateMap was merely holding pointers that were being managed
	// by whoever passed that FSMStateVector to the constructor
	//It's better to have the owner of those pointers do the cleanup, but I suppose you can do that here, too

	//This might be a good place to call EndFSM() if you are not already calling it somewhere else
}

///////////////////////////////////////////////////////////////////////////////
void DeferredFSM::InitFSM( FSMStateVector const& rFiniteStates )
{
	// InitFSM was coded to only be called once per object lifetime
	// to avoid any memory jiggling associated calling m_StateMap.clear()
	// and then filling the map back up again with the same exact stuff
	assert( m_stateMap.empty() && "Hey programmer, InitFSM only needs to be called once per object to avoid jiggling memory" );

	for ( FSMStateVector::const_iterator it=rFiniteStates.begin(); it != rFiniteStates.end(); ++it )
	{
		IFSMStateInterface* const pFiniteState = *it;
		FSMUniqueStateID const uniqueStateID = pFiniteState->GetUniqueStateID();
		assert( ( m_stateMap.end() == m_stateMap.find( uniqueStateID ) ) && "Hey programmer, it looks like you might be trying to insert two states into the machine that share the same ID" );
		m_stateMap.insert( std::make_pair( uniqueStateID, pFiniteState ) );
	}

	assert( ( m_stateMap.end() != m_stateMap.find(GetDefaultRequestID() ) ) && "Hey programmer, make sure your default state is legal, mmmkay?" );

	//Make sure none of the pending state requests become memory leaks
	for (int i = 0; i < kNumStateRequestQueues; ++i)
	{
		for (FSMStateRequest const* pStateRequest : m_pendingStateRequests[i])
		{
			STATEREQUEST_MEMORY_POOL_DELETE(pStateRequest);
		}
		m_pendingStateRequests[i].clear();
	}
	m_activeStateRequestQueueIndex = 0;
	m_pCurrentState = nullptr;

}

///////////////////////////////////////////////////////////////////////////////
void DeferredFSM::BeginFSM()
{
	RequestDefaultState();
}

///////////////////////////////////////////////////////////////////////////////
void DeferredFSM::RequestStateChange(FSMStateRequest const* const pIncomingStateRequest)
{
	assert( ( m_stateMap.end() != m_stateMap.find(pIncomingStateRequest->GetRequestID() ) ) && "Somehow, a state was requested that does not exist within the machine" );

	std::list<FSMStateRequestQueue::iterator> pendingRequestsToRemoveIfNewGuySurvives;

	FSMStateRequestQueue& activeStateRequestQueue = m_pendingStateRequests[m_activeStateRequestQueueIndex];
	
	bool bRequestPending = true;
	for ( FSMStateRequestQueue::iterator it = activeStateRequestQueue.begin(); bRequestPending && (it != activeStateRequestQueue.end()); ++it )
	{
		// Each incoming pending request must pass the gauntlet of all existing pending requests
		// just like the new guy in prison hoping to one day make it as the "current state"
		FSMStateRequest const* const pPendingStateRequest = *it;

		FSMStateRequest::VersusResult const versus = pIncomingStateRequest->VersusOtherRequest( *pPendingStateRequest);
		switch ( versus )
		{
		case FSMStateRequest::VersusResult_BothSurvive:
			break;
		case FSMStateRequest::VersusResult_CancelSelf:
			STATEREQUEST_MEMORY_POOL_DELETE(pIncomingStateRequest);
			bRequestPending = false;
			break;
		case FSMStateRequest::VersusResult_CancelOther:
			pendingRequestsToRemoveIfNewGuySurvives.push_back(it);
			break;
		case FSMStateRequest::VersusResult_CancelBoth:
			it = activeStateRequestQueue.erase(it);
			STATEREQUEST_MEMORY_POOL_DELETE(pPendingStateRequest);
			STATEREQUEST_MEMORY_POOL_DELETE(pIncomingStateRequest);
			bRequestPending = false;
			break;
		}
	}

	if ( bRequestPending )
	{
		for (std::list<FSMStateRequestQueue::iterator>::iterator it = pendingRequestsToRemoveIfNewGuySurvives.begin(); it != pendingRequestsToRemoveIfNewGuySurvives.end(); ++it)
		{
			FSMStateRequest const* const pRequestToDelete = **it;
			activeStateRequestQueue.erase(*it);
			STATEREQUEST_MEMORY_POOL_DELETE(pRequestToDelete);
		}
		
		activeStateRequestQueue.push_back(pIncomingStateRequest);
	}
}

///////////////////////////////////////////////////////////////////////////////
FSMUniqueStateID DeferredFSM::GetCurrentStateID() const
{
	FSMUniqueStateID const currentStateID = ( NULL != m_pCurrentState )
														? m_pCurrentState->GetUniqueStateID()
														: GetDefaultRequestID();
	return currentStateID;
}

///////////////////////////////////////////////////////////////////////////////
// LOGIC BASED ON PENDING STATES CAN LEAD TO SUBTLE BUGS
// THIS FUNCTION IS LEFT COMMENTED TO ACKNOWLEDGE YOUR DESIRE AND WARN
// YOU TO RESIST THE TEMPTATION TO USE PENDING STATE IN YOUR LOGIC
//bool DeferredFSM::IsStateCurrentOrPending( FSMUniqueStateID const uniqueStateID ) const
//{
//	bool bIsCurrentOrPending = ( NULL != m_pCurrentState )
//										&& ( uniqueStateID == m_pCurrentState->GetUniqueStateID() );
//
//	if ( !bIsCurrentOrPending )
//	{
//		for ( FSMStateRequestQueue::const_iterator it = m_PendingStateRequests.begin(); it != m_PendingStateRequests.end(); ++it )
//		{
//			FSMStateRequest const pendingStateRequest = *it;
//			if ( uniqueStateID == pendingStateRequest.GetRequestID() )
//			{
//				bIsCurrentOrPending = true;
//				break;
//			}
//		}
//	}
//
//	return bIsCurrentOrPending;
//}

///////////////////////////////////////////////////////////////////////////////
IFSMStateInterface const* const DeferredFSM::GetStateForQuery( FSMUniqueStateID const uniqueStateID ) const
{
	FSMStateMap::const_iterator const mapIt = m_stateMap.find( uniqueStateID );
	IFSMStateInterface const* const pQueriedState = ( mapIt != m_stateMap.end() )
													? mapIt->second
													: NULL;
	return pQueriedState;
}

///////////////////////////////////////////////////////////////////////////////
void DeferredFSM::TickFSM( float const fDeltaSeconds )
{
	ProcessPendingStates();

	if ( NULL != m_pCurrentState )
	{
		if ( m_pCurrentState->IsStateFinished() )
		{
			IFSMStateInterface* const pFinishedState = m_pCurrentState;
			RequestDefaultState();
			ProcessPendingStates();
			assert( ( pFinishedState != m_pCurrentState ) && "Hey programmer, IsStateFinished() happened, but all transitions got denied" );
			if ( NULL != m_pCurrentState )
			{
				m_pCurrentState->TickState( fDeltaSeconds );
			}
		}
		else
		{
			m_pCurrentState->TickState( fDeltaSeconds );
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
void DeferredFSM::ProcessPendingStates()
{
	FSMStateRequestQueue& processingStateRequestQueue = m_pendingStateRequests[m_activeStateRequestQueueIndex];

	//Make sure all subsequent state transitions go to another queue
	m_activeStateRequestQueueIndex = (1 + m_activeStateRequestQueueIndex) % kNumStateRequestQueues;

	while ( !processingStateRequestQueue.empty() )
	{
		FSMStateRequest const* const pPendingStateRequest = processingStateRequestQueue.front();
		// 1. Ask the owner if this is allowed...
		bool bRequestGranted = IsStateChangeAllowed(*pPendingStateRequest);
		if ( bRequestGranted && ( NULL != m_pCurrentState ) )
		{
			// 2. Next, we ask the current state if this is allowed
			bRequestGranted = m_pCurrentState->IsStateFinished() || !m_pCurrentState->FilterIncomingRequest(*pPendingStateRequest);
		}
		if ( bRequestGranted )
		{
			FSMStateMap::const_iterator const mapIt = m_stateMap.find( pPendingStateRequest->GetRequestID() );
			// There is an assert within DeferredFSM::RequestStateChange that ensures that
			// we have a legal state ID
			IFSMStateInterface* const pPendingState = mapIt->second;

			// 3. Now the state itself gets a chance to be allowed
			bRequestGranted = pPendingState->CanEnterState( *pPendingStateRequest );
			if ( bRequestGranted )
			{
				// Do the state transition HERE...
				if ( NULL != m_pCurrentState )
				{
					m_pCurrentState->DoExitState( pPendingStateRequest );
				}
				assert( ( pPendingState->GetUniqueStateID() == pPendingStateRequest->GetRequestID() ) && "Hey programmer, a promise for unique state ID to match the pending state has somehow been broken" );
				pPendingState->DoEnterState( *pPendingStateRequest, m_pCurrentState );
				m_pCurrentState = pPendingState;
			}
		}

		// Note from Kain: I thought about supporting the concept of "Lingering Requests"
		// where a request would stick around for X seconds trying to become current
		// while it lingered, but then I thought about the prospect of hard to repro
		// bugs that might pop up because a request from X seconds ago lingered and
		// manifested at some inopportune time... and so I backed out because I figured
		// the reward of lingering state requests would not outweigh the risk.
		// Linger state requests should be done by having calling code actively manage that intention
		// instead of doing a fire-and-forget usage pattern

		processingStateRequestQueue.pop_front();
		STATEREQUEST_MEMORY_POOL_DELETE(pPendingStateRequest);
	}
}

///////////////////////////////////////////////////////////////////////////////
void DeferredFSM::EndFSM()
{
	if ( NULL != m_pCurrentState )
	{
		m_pCurrentState->DoExitState( NULL );
		m_pCurrentState = NULL;
	}

	//Make sure none of the pending state requests become memory leaks
	for (int i = 0; i < kNumStateRequestQueues; ++i)
	{
		for (FSMStateRequest const* pStateRequest : m_pendingStateRequests[i])
		{
			STATEREQUEST_MEMORY_POOL_DELETE(pStateRequest);
		}
		m_pendingStateRequests[i].clear();
	}
}
